#ifndef __TAutomaticPointer__
#define __TAutomaticPointer__

#include "TPointerDeleter.hpp"
using Exponent::Collections::TCountedObjectDeleter;

//	===========================================================================
namespace Exponent
{
	namespace Collections
	{
		/**
		 * @class TAutomaticPointer TAutomaticPointer.hpp
		 * @brief An automatic pointer that can be created on the stack to automatically delete any dynamically allocated resources
		 *
		 * In use, this class is very simple. Inside a function, rather than using delete on a variable do something like this\n
		 * @code
		 * void foo()
		 * {
		 *     TAutomaticPointer<CPoint> variable = new CPoint(10, 10);
		 *     // ...
		 *     // Do some processing here
		 *     // ...
		 *     // variable is automatically delete here, which deletes the point too
		 * }
		 * @endcode
		 * Note that you can use a TAutomaticPointer in the same way as a normal pointer\n
		 * @code
		 * void foo()
		 * {
		 *     TAutomaticPointer<CPoint> variable = new CPoint(10, 10);
		 *     const long x = variable->getXPosition();	   // Overloaded arrow operator returns the pointer
		 *     *variable.setXPosition(x + 10);			   // Overloaded deference operator returns deferenced pointer
		 * }
		 * @endcode
		 * It should also be pointed out that this class can use any standard ObjectDeleter, as used by the arrays\n
		 * This means that you can make it operate like both the STL auto_ptr and the Boost smart_ptr in operation\n
		 * STL auto_ptr:\n
		 * @code
		 * TAutomaticPointer<CPoint> ptr(new CPoint(x,y), TObjectDeleter<CPoint>);
		 * @endcode
		 *\n
		 * Boost smart_ptr:\n
		 * @code
		 * TAutomaticPointer<CPoint> variable = new CPoint(10, 10);
		 * @endcode
		 *
		 * @see TCountedObjectDeleter
		 *
		 * @date 03/05/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * @note This class is intended for internal types and counted objects. \n
		 * It will not work for 'built in' c++ types. If you need this functionality you can use any of the items below:
		 * @see Exponent::Basics::CFloat
		 * @see Exponent::Basics::CString
		 * @see Exponent::Basics::CDouble
		 * @see Exponent::Basics::CLong
		 * @see Exponent::Basics::CBool
		 *
		 * $Id: TAutomaticPointer.hpp,v 1.11 2007/02/11 02:15:54 paul Exp $
		 */
		template <class TypeName> class TAutomaticPointer : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			static TCountedObjectDeleter<TypeName> TAUTOMATICPOINTER_DEFAULT_DELETER;			/**< The default handler for pointer deletion */

//	===========================================================================


			/**
			 * Construction
			 * @param pointerDeleter The object used for addition and deletion of pointers, can be NULL for deletion of a pointer as default
			 */
			TAutomaticPointer(TPointerDeleter<TypeName> *pointerDeleter = &TAUTOMATICPOINTER_DEFAULT_DELETER) : m_pointer(NULL), m_pointerDeletionHandler(NULL)
			{
				EXPONENT_CLASS_CONSTRUCTION(TAutomaticPointer<TypeName>);
				NULL_POINTER(m_pointer);
				m_pointerDeletionHandler = pointerDeleter;
			}

			/**
			 * Construction
			 * @param pointer The pointer to store
			 * @param pointerDeleter The object used for addition and deletion of pointers, can be NULL for deletion of a pointer as default
			 */
			TAutomaticPointer(TypeName *pointer, TPointerDeleter<TypeName> *pointerDeleter = &TAUTOMATICPOINTER_DEFAULT_DELETER) : m_pointer(NULL), m_pointerDeletionHandler(NULL)
			{
				EXPONENT_CLASS_CONSTRUCTION(TAutomaticPointer<TypeName>);

				// If valid store the pointer
				if (pointer)
				{
					m_pointer = pointer;
				}

				// STore the deleter
				m_pointerDeletionHandler = pointerDeleter;

				// If we have, then we want to notify the handler
				if (m_pointerDeletionHandler && m_pointer)
				{
					m_pointerDeletionHandler->pointerAdded(m_pointer);
				}
			}

			/**
			 * Copy Construction
			 * @param pointer The pointer to store
			 * @param pointerDeleter The object used for addition and deletion of pointers, can be NULL for deletion of a pointer as default
			 */
			TAutomaticPointer(const TAutomaticPointer<TypeName> &pointer, TPointerDeleter<TypeName> *pointerDeleter = &TAUTOMATICPOINTER_DEFAULT_DELETER) : m_pointer(NULL), m_pointerDeletionHandler(NULL)
			{
				EXPONENT_CLASS_CONSTRUCTION(TAutomaticPointer<TypeName>);
				NULL_POINTER(m_pointer);
				m_pointerDeletionHandler = pointerDeleter;
				*this = pointer;
			}

			/**
			 * Destruction
			 */
			virtual ~TAutomaticPointer()
			{
				EXPONENT_CLASS_DESTRUCTION(TAutomaticPointer<TypeName>);
				this->freePointer();
			}

//	===========================================================================

			/**
			 * Assignment operator
			 * @param pointer The pointer to store
			 * @retval TAutomaticPointer<TypeName>& A reference to this
			 */
			TAutomaticPointer<TypeName> &operator = (const TAutomaticPointer<TypeName> &pointer)
			{
				// Do we have one currently
				if (m_pointer)
				{
					// First we have to delete the old one
					if (m_pointerDeletionHandler)
					{
						// Delete it
						m_pointerDeletionHandler->deletePointer(m_pointer);
						NULL_POINTER(m_pointer);
					}
					else
					{
						FREE_POINTER(m_pointer);
					}
				}

				// Now we copy the new one
				m_pointer = pointer.m_pointer;

				// If we have, then we want to notify the handler
				if (m_pointerDeletionHandler && m_pointer)
				{
					m_pointerDeletionHandler->pointerAdded(m_pointer);
				}

				return *this;
			}


			/**
			 * Assignment operator
			 * @param pointer The pointer to store
			 * @retval TAutomaticPointer<TypeName>& A reference to this
			 */
			TAutomaticPointer<TypeName> &operator = (TypeName *pointer)
			{
				// Do we have one currently
				if (m_pointer)
				{
					// First we have to delete the old one
					if (m_pointerDeletionHandler)
					{
						// Delete it
						m_pointerDeletionHandler->deletePointer(m_pointer);
						NULL_POINTER(m_pointer);
					}
					else
					{
						FREE_POINTER(m_pointer);
					}
				}

				// Now we copy the new one
				if (pointer)
				{
					m_pointer = pointer;
				}

				// If we have, then we want to notify the handler
				if (m_pointerDeletionHandler && m_pointer)
				{
					m_pointerDeletionHandler->pointerAdded(m_pointer);
				}

				return *this;
			}

//	===========================================================================

			/**
			 * Pointer operator
			 * @retval TypeName* A pointer to the innner data
			 */
			TypeName *operator -> () { return m_pointer; }

			/**
			 * Deference operator
			 * @retval TypeName& A reference to the inner pointer
			 */
			TypeName &operator * () { return *m_pointer; }

			/**
			 * Is the pointer valid
			 * @retval bool True if pointer is non NULL, false otherwise
			 */
			bool pointerIsValid() const { return m_pointer != NULL; }

			/**
			 * Free the pointer held internally, called automatically at destruction
			 */
			void freePointer()
			{
				// Ifwe have a handler
				if (m_pointerDeletionHandler)
				{
					// Delete it
					m_pointerDeletionHandler->deletePointer(m_pointer);
				}
				else
				{
					FREE_POINTER(m_pointer);
				}
				NULL_POINTER(m_pointer);
			}

			/**
			 * Get the pointer
			 * @retval const TypeName* The pointer
			 */
			const TypeName *getPointer() const { return m_pointer; }

			/**
			 * Get the pointer
			 * @retval TypeName* The pointer
			 */
			TypeName *getMutablePointer() { return m_pointer; }

//	===========================================================================

		protected:

//	===========================================================================

			TypeName *m_pointer;										/**< The pointer object */
			TPointerDeleter<TypeName> *m_pointerDeletionHandler;		/**< Deletion handler */

		};

		/** @cond */
		EXPONENT_TEMPLATE_CLASS_IMPLEMENTATION(TAutomaticPointer<TypeName>, TypeName, CCountedObject);
		template<class TypeName> TCountedObjectDeleter<TypeName> TAutomaticPointer<TypeName>::TAUTOMATICPOINTER_DEFAULT_DELETER;
		/** @endcond */
	}
}
#endif
